var application = null;

function startApplication()
{
    application = new Appliction();

    application.addRetroPlayerObservers();
    application.startPositionUpdate();
};

function timeIntervalToString(timeInterval) {
    var minutes = Math.floor(timeInterval / 60);
    var seconds = Math.floor(timeInterval) % 60;

    return "" + minutes + ":" + ((seconds >= 10) ? "" : "0") + seconds;        
}

class Control
{
    constructor(element)
    {
        this.element = element;
        this.visibleDisplayStyle = element.style.display;
    }

    setVisible(newVisible)
    {
        this.element.style.display = newVisible ? this.visibleDisplayStyle : "none";
    }

    setClassEnabled(className, classEnabled)
    {
        if(classEnabled)
        {
            this.element.classList.add(className);
        }
        else
        {
            this.element.classList.remove(className);
        }        
    }

    setEnabled(newEnabled)
    {
        this.setClassEnabled("disabled", !newEnabled);
    }

    setHighlighted(newHighlighted)
    {
        this.setClassEnabled("highlighted", newHighlighted);
    }

    setClickHandler(newClickHandler)
    {
        var clickBox = this.element.getElementsByClassName("clickBox")[0];
        
        if(clickBox)
        {
            clickBox.onclick = newClickHandler;
        }
        else
        {
            this.element.onclick = newClickHandler;
        }
    }

    setImage(newImageUrl)
    {
        let imageElement = this.element.getElementsByTagName("img")[0];

        if(imageElement)
        {
            imageElement.src = newImageUrl;
        }
    }
}

class ProgressBar extends Control
{
    constructor(progressBarElement)
    {
        super(progressBarElement);

        this.fillElement = document.createElement("div");
        this.element.appendChild(this.fillElement);
    }

    setProgress(progress)
    {
        var progressBarStyle = window.getComputedStyle(this.element);

        var paddingLeft   = parseFloat(progressBarStyle.paddingLeft);
        var paddingRight  = parseFloat(progressBarStyle.paddingRight);
        var paddingTop    = parseFloat(progressBarStyle.paddingTop);
        var paddingBottom = parseFloat(progressBarStyle.paddingBottom);

        var contentWidth  = this.element.clientWidth - paddingLeft - paddingRight;
        var contentHeight = this.element.clientHeight - paddingTop  - paddingBottom;

        var progressWidth = Math.floor(contentWidth * progress);

        this.fillElement.style.width       = "" + progressWidth + "px";
        this.fillElement.style.height      = "" + contentHeight + "px";
        this.fillElement.style.marginRight = "" + (contentWidth - progressWidth) + "px";
    }
}

class Label extends Control
{
    constructor(labelElement)
    {
        super(labelElement);
    }

    setText(newText)
    {
        this.element.textContent = newText ? newText : "";
    }
}

class Appliction
{
    constructor()
    {
        this.htmlElement      = document.getElementsByTagName("html")[0];
        this.bodyElement      = document.getElementsByTagName("body")[0];
        this.imageElement     = document.getElementById("image");
        this.controlsElement  = document.getElementById("controls");
        this.titleLabel       = new Label(document.getElementById("songName"));
        this.artistLabel      = new Label(document.getElementById("artist"));
        this.positionLabel    = new Label(document.getElementById("position"));
        this.durationLabel    = new Label(document.getElementById("duration"));
        this.previousButton   = new Control(document.getElementById("previousButton"));
        this.playButton       = new Control(document.getElementById("playButton"));
        this.pauseButton      = new Control(document.getElementById("pauseButton"));
        this.nextButton       = new Control(document.getElementById("nextButton"));
        this.repeatAllButton  = new Control(document.getElementById("repeatAllButton"));
        this.repeatSongButton = new Control(document.getElementById("repeatSongButton"));
        this.shuffleButton    = new Control(document.getElementById("shuffleButton"));        
        this.progressBar      = new ProgressBar(document.getElementById("progressBar"));

        var self = this;

        this.repeatAllButton.setClickHandler(function () { self.cycleRepeat(); });
        this.repeatSongButton.setClickHandler(function () { self.cycleRepeat(); });
        this.shuffleButton.setClickHandler(function () { self.toggleShuffleEnabled(); });

        this.imageElement.addEventListener("load", function () { self.adjustWindowSize(); });
    }

    startPositionUpdate()
    {
        var self = this;
        window.setTimeout(function() {
            RetroPlayer.updateProperties(["position"]);
            self.startPositionUpdate();
        }, 100);
    }

    updateProgress(position, duration)
    {
        this.positionLabel.setText(timeIntervalToString(position));
        this.durationLabel.setText(duration > 0 ? timeIntervalToString(duration) : "");

        this.progressBar.setProgress(position >= duration ? 1 : position / duration);
    }

    stateChanged(newState)
    {
        this.pauseButton.setVisible(RetroPlayer.isPlaying());
        this.playButton.setVisible(!RetroPlayer.isPlaying());
    }

    currentSongChanged(newCurrentSong)
    {
        if(newCurrentSong)
        {
            if(newCurrentSong.imageFilePath)
            {
                this.imageElement.setAttribute("src", newCurrentSong.imageFilePath);
                this.imageElement.style.display = "block";
            }
            else
            {
                this.imageElement.style.display = "none";            
            }

            this.titleLabel.setText(newCurrentSong.name);
            this.artistLabel.setText(newCurrentSong.artist);
        }
        else
        {
            this.imageElement.style.display = "none";
            this.titleLabel.setText("");
            this.artistLabel.setText("");        
        }

        this.adjustWindowSize()
    }

    adjustWindowSize()
    {
        var controlsMarginTop = 0;

        var windowWidth  = this.controlsElement.clientWidth;
        var windowHeight = this.controlsElement.clientHeight;

        var imageWidth  = this.imageElement.clientWidth;
        var imageHeight = this.imageElement.clientHeight;

        if(this.imagePosition == "top")
        {
            windowHeight += imageHeight;

            if(windowWidth < imageWidth)
            {
                windowWidth = imageWidth;
            }
        }
        else
        {
            windowWidth += imageWidth;

            if(windowHeight < imageHeight)
            {
                controlsMarginTop = Math.floor((imageHeight - windowHeight) / 2);

                windowHeight = imageHeight;
            }
        }

        this.controlsElement.style.marginTop = controlsMarginTop + "px";

        var bodyStyle = window.getComputedStyle(this.bodyElement);

        windowWidth  += parseInt(bodyStyle.marginLeft) + parseInt(bodyStyle.marginRight);
        windowHeight += parseInt(bodyStyle.marginTop)  + parseInt(bodyStyle.marginBottom);

        RetroPlayer.postCommand("setWindowStyle", { "windowStyle" : { 
            "height" : "" + windowHeight,
            "width"  : "" + windowWidth }});    
    }

    playQueueChanged(newPlayQueue)
    {
        this.playButton.setEnabled(newPlayQueue.songCount > 0);
        this.pauseButton.setEnabled(newPlayQueue.songCount > 0);

        this.nextButton.setEnabled(newPlayQueue.hasNextSong);
        this.previousButton.setEnabled(newPlayQueue.hasPreviousSong);
    }

    systemAppearanceChanged(newAppearance)
    {
        this.systemAppearance = newAppearance;

        this.updateAppearance();
    }

    updateAppearance()
    {
        var userAppearance = RetroPlayer.getUserDefault("appearance", "");

        if(userAppearance.length > 0)
        {
            this.htmlElement.className = userAppearance;            
        }
        else
        {
            this.htmlElement.className = this.systemAppearance;            
        }

        RetroPlayer.setWindowStyle(window.getComputedStyle(document.documentElement));
    }

    updateImagePosition()
    {
        this.imagePosition = RetroPlayer.getUserDefault("imagePosition", "left");

        this.imageElement.classList.remove("top");
        this.imageElement.classList.remove("left");
        this.imageElement.classList.remove("right");

        this.imageElement.classList.add(this.imagePosition);
    }

    userDefaultsChanged()
    {
        this.updateAppearance();
        this.updateImagePosition();
        this.adjustWindowSize()
    }

    cycleRepeat()
    {
        if(RetroPlayer.repeat == "off")
        {
            RetroPlayer.setRepeat("song");
        }
        else if(RetroPlayer.repeat == "song")
        {
            RetroPlayer.setRepeat("all");
        }
        else if(RetroPlayer.repeat == "all")
        {
            RetroPlayer.setRepeat("off");
        }    
    }

    repeatChanged(newRepeat)
    {
        if(newRepeat == "off")
        {
            this.repeatAllButton.setHighlighted(false);
            this.repeatAllButton.setVisible(true);
            this.repeatSongButton.setHighlighted(false);
            this.repeatSongButton.setVisible(false);
        }
        else if(newRepeat == "song")
        {
            this.repeatAllButton.setHighlighted(false);
            this.repeatAllButton.setVisible(false);
            this.repeatSongButton.setHighlighted(true);
            this.repeatSongButton.setVisible(true);
        }
        else if(newRepeat == "all")
        {
            this.repeatAllButton.setHighlighted(true);
            this.repeatAllButton.setVisible(true);
            this.repeatSongButton.setHighlighted(false);
            this.repeatSongButton.setVisible(false);
        }
    }

    toggleShuffleEnabled()
    {
        RetroPlayer.setShuffleEnabled(!RetroPlayer.shuffleEnabled);
    }

    shuffleEnabledChanged(newShuffleEnabled)
    {
        this.shuffleButton.setHighlighted(newShuffleEnabled);
    }

    addRetroPlayerObservers()
    {
        var self = this;
        RetroPlayer.addPropertyObserver("state", function() {
            self.stateChanged(RetroPlayer.state);
        });

        RetroPlayer.addPropertyObserver("currentSong", function() {
            self.currentSongChanged(RetroPlayer.currentSong);
        });

        RetroPlayer.addPropertyObserver("playQueue", function() {
            self.playQueueChanged(RetroPlayer.playQueue);
        });

        RetroPlayer.addPropertyObserver("repeat", function() {
            self.repeatChanged(RetroPlayer.repeat);
        });

        RetroPlayer.addPropertyObserver("shuffleEnabled", function() {
            self.shuffleEnabledChanged(RetroPlayer.shuffleEnabled);
        });

        RetroPlayer.addPropertyObserver("position", function() {
            self.updateProgress(RetroPlayer.position, RetroPlayer.currentSong ? RetroPlayer.currentSong.duration : 0);
        });

        RetroPlayer.addPropertyObserver("userDefaults", function() {
            self.userDefaultsChanged();
        });

        RetroPlayer.addPropertyObserver("appearance", function() {
            self.systemAppearanceChanged(RetroPlayer.appearance);
        });

        RetroPlayer.updateProperties([ "appearance" ]);        
    }
}
